<?php

/*
	File: gbtxEvent.inc.php

	Definition of the gbtx Event object.

	Title: gbtxEvent

	Please see <copyright.inc.php> for a detailed description, copyright
	and license information.
*/

/*
	@package gbtx
	@version $Id: gbtxEvent.inc.php 362 2007-05-29 15:32:24Z calltoconstruct $
	@copyright Copyright (c) 2005-2006 by Jared White & J. Max Wilson
	@license http://www.gbtxproject.org/bsd_license.txt BSD License
*/

// require_once is necessary here as the function plugin also includes this
//SkipAIO
require_once dirname(__FILE__) . '/gbtxUserFunction.inc.php';
//EndSkipAIO

/*
	Class: gbtxEvent
	
	A container class which holds a reference to handler functions and configuration
	options associated with a registered event.
*/
class gbtxEvent
{
	/*
		String: sName
		
		The name of the event.
	*/
	var $sName;
	
	/*
		Array: aConfiguration
		
		Configuration / call options to be used when initiating a gbtx request
		to trigger this event.
	*/
	var $aConfiguration;
	
	/*
		Array: aHandlers
		
		A list of <gbtxUserFunction> objects associated with this registered
		event.  Each of these functions will be called when the event is triggered.
	*/
	var $aHandlers;
	
	/*
		Function: gbtxEvent
		
		Construct and initialize this <gbtxEvent> object.
	*/
	function gbtxEvent($sName)
	{
		$this->sName = $sName;
		$this->aConfiguration = array();
		$this->aHandlers = array();
	}
	
	/*
		Function: getName
		
		Returns the name of the event.
		
		Returns:
		
		string - the name of the event.
	*/
	function getName()
	{
		return $this->sName;
	}
	
	/*
		Function: configure
		
		Sets/stores configuration options that will be used when generating
		the client script that is sent to the browser.
	*/
	function configure($sName, $mValue)
	{
		$this->aConfiguration[$sName] = $mValue;
	}
	
	/*
		Function: addHandler
		
		Adds a <gbtxUserFunction> object to the list of handlers that will
		be fired when the event is triggered.
	*/
	function addHandler(&$xuf)
	{
		$this->aHandlers[] =& $xuf;
	}
	
	/*
		Function: generateRequest
		
		Generates a <gbtxRequest> object that corresponds to the
		event so that the client script can easily invoke this event.
		
		sgbtxPrefix - (string):  The prefix that will be prepended to
			the client script stub function associated with this event.
			
		sEventPrefix - (string):  The prefix prepended to the client script
			function stub and <gbtxRequest> script.
	*/
	function generateRequest($sgbtxPrefix, $sEventPrefix)
	{
		$sEvent = $this->sName;
		return new gbtxRequest("{$sgbtxPrefix}{$sEventPrefix}{$sEvent}");
	}
 	
 	/*
 		Function: generateClientScript
 		
 		Generates a block of javascript code that declares a stub function
 		that can be used to easily trigger the event from the browser.
 	*/
 	function generateClientScript($sgbtxPrefix, $sEventPrefix)
	{
		$sMode = '';
		$sMethod = '';
		
		if (isset($this->aConfiguration['mode']))
			$sMode = $this->aConfiguration['mode'];
			
		if (isset($this->aConfiguration['method']))
			$sMethod = $this->aConfiguration['method'];
			
		if (0 < strlen($sMode))
			$sMode = ", mode: '{$sMode}'";
		
		if (0 < strlen($sMethod))
			$sMethod = ", method: '{$sMethod}'";
		
		$sEvent = $this->sName;
		echo "{$sgbtxPrefix}{$sEventPrefix}{$sEvent} = function() { return gbtx.request( { xjxevt: '{$sEvent}' }, { parameters: arguments{$sMode}{$sMethod} } ); };\n";
	}
	
	/*
		Function: fire
		
		Called by the <gbtxEventPlugin> when the event has been triggered.
	*/
	function fire($aArgs)
	{
		$objResponseManager =& gbtxResponseManager::getInstance();
		
		foreach (array_keys($this->aHandlers) as $sKey)
			$this->aHandlers[$sKey]->call($aArgs);
	}
}
